<?php
/*
Plugin Name: Type With AI
Description: AI writing assistant inside WordPress editor. Free mode: 10 requests/day. Pro mode: Unlimited with an active license key (enforced on your proxy).
Version: 2.0.0
Author: Type With AI
Text Domain: type-with-ai
*/

if ( ! defined('ABSPATH') ) exit;

class Type_With_AI {

    const OPT_LICENSE_KEY = 'twa_license_key';
    const API_URL = 'https://type-with-ai.com/api/ai-proxy.php';

    public function __construct() {
        add_action('admin_menu', [$this, 'settings_page']);
        add_action('admin_init', [$this, 'register_settings']);

        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('wp_ajax_twa_generate', [$this, 'ajax_generate']);
    }

    public function settings_page() {
        add_options_page(
            __('Type With AI Settings', 'type-with-ai'),
            __('Type With AI', 'type-with-ai'),
            'manage_options',
            'type-with-ai-settings',
            [$this, 'render_settings']
        );
    }

    public function register_settings() {
        register_setting('twa_settings', self::OPT_LICENSE_KEY, [
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => '',
        ]);

        add_settings_section('twa_section', __('Connection', 'type-with-ai'), '__return_false', 'type-with-ai-settings');

        add_settings_field('twa_license', __('License Key (optional)', 'type-with-ai'), function() {
            $val = esc_attr(get_option(self::OPT_LICENSE_KEY, ''));
            echo '<input type="text" name="'.esc_attr(self::OPT_LICENSE_KEY).'" value="'.$val.'" style="width:420px" placeholder="PRO-XXXX-XXXX-XXXX" />';
            echo '<p class="description">Leave empty for Free (10 requests/day). Enter an active key for Pro (unlimited). If the key is invalid, the system will fall back to Free.</p>';
        }, 'type-with-ai-settings', 'twa_section');
    }

    public function render_settings() {
        if (!current_user_can('manage_options')) return;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Type With AI Settings', 'type-with-ai'); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('twa_settings'); ?>
                <?php do_settings_sections('type-with-ai-settings'); ?>
                <?php submit_button(); ?>
            </form>

            <hr>
            <h2><?php esc_html_e('Plans', 'type-with-ai'); ?></h2>
            <ul>
                <li><strong><?php esc_html_e('Free:', 'type-with-ai'); ?></strong> <?php esc_html_e('10 AI requests/day. No key required.', 'type-with-ai'); ?></li>
                <li><strong><?php esc_html_e('Pro:', 'type-with-ai'); ?></strong> <?php esc_html_e('Unlimited requests with an active license key.', 'type-with-ai'); ?></li>
            </ul>
            <p><a href="https://type-with-ai.com" target="_blank" rel="noopener"><?php esc_html_e('Manage or buy a license', 'type-with-ai'); ?></a></p>
        </div>
        <?php
    }

    public function add_meta_box() {
        $post_types = ['post','page'];
        if (post_type_exists('product')) $post_types[] = 'product';

        foreach ($post_types as $pt) {
            add_meta_box('twa_box', __('Type With AI', 'type-with-ai'), [$this, 'render_box'], $pt, 'side', 'high');
        }
    }

    private function site_fingerprint() : string {
        // Used to count free usage when license key is empty or invalid.
        $site = site_url();
        return 'FREE-' . strtoupper(substr(hash('sha256', $site), 0, 12));
    }

    public function render_box($post) {
        $nonce = wp_create_nonce('twa_nonce');
        $post_title = get_the_title($post);

        $license = trim((string) get_option(self::OPT_LICENSE_KEY, ''));
        $has_license = $license !== '';
        ?>
        <div id="twa-container">
            <p style="margin-top:0;">
                <strong><?php esc_html_e('Plan:', 'type-with-ai'); ?></strong>
                <?php echo $has_license ? esc_html__('Pro key entered (validation happens on server)', 'type-with-ai') : esc_html__('Free (no key)', 'type-with-ai'); ?>
            </p>

            <p><strong><?php esc_html_e('Mode', 'type-with-ai'); ?></strong><br>
                <select id="twa_mode" style="width:100%;">
                    <option value="generate"><?php esc_html_e('Generate', 'type-with-ai'); ?></option>
                    <option value="rewrite"><?php esc_html_e('Rewrite', 'type-with-ai'); ?></option>
                    <option value="summarize"><?php esc_html_e('Summarize', 'type-with-ai'); ?></option>
                    <option value="seo_title"><?php esc_html_e('SEO Title (Pro)', 'type-with-ai'); ?></option>
                    <option value="seo_description"><?php esc_html_e('SEO Description (Pro)', 'type-with-ai'); ?></option>
                    <option value="product_description"><?php esc_html_e('Product Description (Pro)', 'type-with-ai'); ?></option>
                </select>
                <?php if (!$has_license): ?>
                    <span style="display:block;margin-top:6px;font-size:11px;color:#666;">
                        Pro modes require an active license. <a href="https://type-with-ai.com" target="_blank" rel="noopener">Upgrade</a>
                    </span>
                <?php endif; ?>
            </p>

            <p><strong><?php esc_html_e('Prompt / Instructions', 'type-with-ai'); ?></strong>
                <textarea id="twa_prompt" style="width:100%;height:80px;" placeholder="Ex: Write a friendly intro about..."></textarea>
            </p>

            <p>
                <button type="button" class="button button-primary" id="twa_btn"><?php esc_html_e('Generate with AI', 'type-with-ai'); ?></button>
            </p>

            <div id="twa_status" style="margin:6px 0;font-size:11px;color:#555;"></div>

            <p><strong><?php esc_html_e('AI Output', 'type-with-ai'); ?></strong>
                <textarea id="twa_out" style="width:100%;height:150px;" placeholder="AI result will appear here..."></textarea>
            </p>

            <p>
                <button type="button" class="button" id="twa_insert"><?php esc_html_e('Insert into editor', 'type-with-ai'); ?></button>
            </p>
        </div>

        <script>
        (function($){
            $(function(){
                const nonce = <?php echo wp_json_encode($nonce); ?>;
                const postTitle = <?php echo wp_json_encode($post_title); ?>;

                function getSelectionText(){
                    try { return window.getSelection ? window.getSelection().toString() : ''; } catch(e){ return ''; }
                }

                $('#twa_btn').on('click', function(e){
                    e.preventDefault();

                    const mode = $('#twa_mode').val() || 'generate';
                    const prompt = $('#twa_prompt').val() || '';
                    const ctx = getSelectionText();

                    $('#twa_status').text('Asking AI...');
                    $('#twa_btn').prop('disabled', true);

                    $.post(ajaxurl, {
                        action: 'twa_generate',
                        nonce: nonce,
                        mode: mode,
                        prompt: prompt,
                        context: ctx,
                        post_title: postTitle
                    }).done(function(resp){
                        $('#twa_btn').prop('disabled', false);
                        if (resp && resp.success) {
                            $('#twa_out').val(resp.data.text || '');
                            $('#twa_status').text('Done ✔');
                        } else {
                            const msg = resp && resp.data && resp.data.message ? resp.data.message : 'Unknown error';
                            $('#twa_status').text('Error: ' + msg);
                        }
                    }).fail(function(xhr){
                        $('#twa_btn').prop('disabled', false);
                        $('#twa_status').text('Request failed. Check proxy URL / server.');
                    });
                });

                $('#twa_insert').on('click', function(e){
                    e.preventDefault();
                    const text = $('#twa_out').val();
                    if (!text) { alert('No AI output to insert.'); return; }

                    // Gutenberg
                    if (typeof wp !== 'undefined' && wp.blocks && wp.blocks.createBlock && wp.data) {
                        try {
                            const block = wp.blocks.createBlock('core/paragraph', { content: text });
                            let disp = null;
                            try { disp = wp.data.dispatch('core/block-editor'); } catch(e1){ disp = null; }
                            if (!disp) { try { disp = wp.data.dispatch('core/editor'); } catch(e2){ disp = null; } }
                            if (disp && disp.insertBlocks) { disp.insertBlocks(block); return; }
                        } catch(err) {}
                    }

                    // Classic editor
                    if (typeof tinymce !== 'undefined' && tinymce.activeEditor && !tinymce.activeEditor.isHidden()) {
                        tinymce.activeEditor.focus();
                        tinymce.activeEditor.selection.setContent(tinymce.activeEditor.selection.getContent() + "\n\n" + text);
                        return;
                    }

                    // Fallback textarea
                    const $content = $('#content');
                    if ($content.length) $content.val($content.val() + "\n\n" + text);
                    else alert('Could not find an editor to insert into.');
                });
            });
        })(jQuery);
        </script>
        <?php
    }

    public function ajax_generate() {
        if (!current_user_can('edit_posts')) wp_send_json_error(['message' => 'Not allowed.']);

        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
        if (!wp_verify_nonce($nonce, 'twa_nonce')) wp_send_json_error(['message' => 'Security check failed.']);

        $proxy = self::API_URL;

        $license = trim((string) get_option(self::OPT_LICENSE_KEY, ''));
        $fingerprint = $this->site_fingerprint(); // used for free usage tracking

        $mode = isset($_POST['mode']) ? sanitize_text_field(wp_unslash($_POST['mode'])) : 'generate';
        $prompt = isset($_POST['prompt']) ? sanitize_textarea_field(wp_unslash($_POST['prompt'])) : '';
        $context = isset($_POST['context']) ? sanitize_textarea_field(wp_unslash($_POST['context'])) : '';
        $post_title = isset($_POST['post_title']) ? sanitize_text_field(wp_unslash($_POST['post_title'])) : '';

        $payload = [
            'license_key' => $license,        // may be empty or invalid; proxy should fall back to Free
            'site_fingerprint' => $fingerprint,
            'mode' => $mode,
            'prompt' => $prompt,
            'context' => $context,
            'post_title' => $post_title,
            'site' => site_url(),
            'plugin' => 'single',
        ];

        $resp = wp_remote_post($proxy, [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => wp_json_encode($payload),
            'timeout' => 30,
        ]);

        if (is_wp_error($resp)) {
            wp_send_json_error(['message' => 'Proxy error: '.$resp->get_error_message()]);
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        if ($code < 200 || $code >= 300 || empty($data)) {
            $msg = is_array($data) && isset($data['message']) ? $data['message'] : ('HTTP '.$code);
            wp_send_json_error(['message' => 'Proxy API error: '.$msg]);
        }

        if (empty($data['success']) || empty($data['text'])) {
            $msg = isset($data['message']) ? $data['message'] : 'Unknown error';
            wp_send_json_error(['message' => $msg]);
        }

        wp_send_json_success(['text' => $data['text']]);
    }
}

new Type_With_AI();
